import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    // Get all tickets with related data
    const tickets = await prisma.ticket.findMany({
      where: { deletedAt: null },
      include: {
        assignedTo: {
          select: {
            id: true,
            name: true,
            email: true,
          },
        },
        normalizedDepartment: {
          select: {
            name: true,
          },
        },
      },
    });

    // Calculate ticket statistics
    const totalTickets = tickets.length;
    console.log("totalTickets", tickets);
    const openTickets = tickets.filter(
      (t) => t.lastKnownStatus === null
    ).length;
    const inProgressTickets = tickets.filter(
      (t) => t.lastKnownStatus === 1
    ).length;
    const completedTickets = tickets.filter(
      (t) => t.lastKnownStatus === 2
    ).length;
    const closedTickets = tickets.filter((t) => t.lastKnownStatus === 3).length;

    // Calculate average resolution time for completed/closed tickets
    const completedTicketsWithTime = tickets.filter(
      (t) =>
        (t.lastKnownStatus === 2 || t.lastKnownStatus === 3) &&
        t.startTime &&
        t.endTime
    );

    const totalResolutionTime = completedTicketsWithTime.reduce(
      (sum, ticket) => {
        if (ticket.startTime && ticket.endTime) {
          return sum + (ticket.endTime.getTime() - ticket.startTime.getTime());
        }
        return sum;
      },
      0
    );

    const averageResolutionTime =
      completedTicketsWithTime.length > 0
        ? totalResolutionTime /
          completedTicketsWithTime.length /
          (1000 * 60 * 60) // Convert to hours
        : 0;

    // Department statistics
    const departmentMap = new Map<
      string,
      {
        count: number;
        totalSeverity: number;
        resolutionTimes: number[];
        elementMap: Map<string | null, number>; // elementId -> count
      }
    >();

    tickets.forEach((ticket) => {
      const dept = ticket.normalizedDepartment?.name || ticket.department;
      if (!departmentMap.has(dept)) {
        departmentMap.set(dept, {
          count: 0,
          totalSeverity: 0,
          resolutionTimes: [],
          elementMap: new Map(),
        });
      }
      const deptData = departmentMap.get(dept)!;
      deptData.count++;
      deptData.totalSeverity += ticket.severity;

      // Track tickets by element for duplicate detection
      const elementId = ticket.elementId;
      deptData.elementMap.set(
        elementId,
        (deptData.elementMap.get(elementId) || 0) + 1
      );

      if (
        ticket.startTime &&
        ticket.endTime &&
        (ticket.lastKnownStatus === 2 || ticket.lastKnownStatus === 3)
      ) {
        const resolutionTime =
          (ticket.endTime.getTime() - ticket.startTime.getTime()) /
          (1000 * 60 * 60);
        deptData.resolutionTimes.push(resolutionTime);
      }
    });

    // Calculate duplicate statistics per department
    const departmentStats = Array.from(departmentMap.entries())
      .map(([department, data]) => {
        // Count duplicates: tickets where same department+element has multiple tickets
        let duplicateCount = 0;
        data.elementMap.forEach((count, elementId) => {
          if (elementId !== null && count > 1) {
            // If element has 2+ tickets, count the extras as duplicates
            duplicateCount += count - 1;
          }
        });

        // Calculate duplicate percentage
        const duplicatePercentage =
          data.count > 0 ? (duplicateCount / data.count) * 100 : 0;

        return {
          department,
          ticketCount: data.count,
          avgSeverity: data.count > 0 ? data.totalSeverity / data.count : 0,
          avgResolutionTime:
            data.resolutionTimes.length > 0
              ? data.resolutionTimes.reduce((sum, time) => sum + time, 0) /
                data.resolutionTimes.length
              : 0,
          duplicateCount,
          duplicatePercentage: Math.round(duplicatePercentage * 10) / 10, // Round to 1 decimal
        };
      })
      .sort((a, b) => b.ticketCount - a.ticketCount);

    // Severity statistics
    const severityMap = new Map<
      number,
      { count: number; resolutionTimes: number[] }
    >();

    tickets.forEach((ticket) => {
      if (!severityMap.has(ticket.severity)) {
        severityMap.set(ticket.severity, { count: 0, resolutionTimes: [] });
      }
      const severityData = severityMap.get(ticket.severity)!;
      severityData.count++;

      if (
        ticket.startTime &&
        ticket.endTime &&
        (ticket.lastKnownStatus === 2 || ticket.lastKnownStatus === 3)
      ) {
        const resolutionTime =
          (ticket.endTime.getTime() - ticket.startTime.getTime()) /
          (1000 * 60 * 60);
        severityData.resolutionTimes.push(resolutionTime);
      }
    });

    const severityStats = Array.from(severityMap.entries())
      .map(([severity, data]) => ({
        severity,
        count: data.count,
        avgResolutionTime:
          data.resolutionTimes.length > 0
            ? data.resolutionTimes.reduce((sum, time) => sum + time, 0) /
              data.resolutionTimes.length
            : 0,
      }))
      .sort((a, b) => a.severity - b.severity);

    // Technician statistics
    const technicianMap = new Map<
      string,
      {
        name: string;
        totalTickets: number;
        completedTickets: number;
        resolutionTimes: number[];
      }
    >();

    tickets.forEach((ticket) => {
      if (ticket.assignedToId && ticket.assignedTo) {
        const techId = ticket.assignedToId;
        if (!technicianMap.has(techId)) {
          technicianMap.set(techId, {
            name: ticket.assignedTo.name || ticket.assignedTo.email,
            totalTickets: 0,
            completedTickets: 0,
            resolutionTimes: [],
          });
        }
        const techData = technicianMap.get(techId)!;
        techData.totalTickets++;

        if (ticket.lastKnownStatus === 2 || ticket.lastKnownStatus === 3) {
          techData.completedTickets++;
          if (ticket.startTime && ticket.endTime) {
            const resolutionTime =
              (ticket.endTime.getTime() - ticket.startTime.getTime()) /
              (1000 * 60 * 60);
            techData.resolutionTimes.push(resolutionTime);
          }
        }
      }
    });

    const technicianStats = Array.from(technicianMap.entries())
      .map(([technicianId, data]) => ({
        technicianId,
        technicianName: data.name,
        totalTickets: data.totalTickets,
        avgResolutionTime:
          data.resolutionTimes.length > 0
            ? data.resolutionTimes.reduce((sum, time) => sum + time, 0) /
              data.resolutionTimes.length
            : 0,
        completionRate:
          data.totalTickets > 0
            ? (data.completedTickets / data.totalTickets) * 100
            : 0,
      }))
      .sort((a, b) => b.totalTickets - a.totalTickets);

    // Time-based statistics
    const now = new Date();
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const weekAgo = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
    const monthAgo = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000);

    const todayTickets = tickets.filter((t) => t.createdAt >= today).length;
    const weekTickets = tickets.filter((t) => t.createdAt >= weekAgo).length;
    const monthTickets = tickets.filter((t) => t.createdAt >= monthAgo).length;
    const avgTicketsPerDay = monthTickets / 30;

    const timeStats = {
      todayTickets,
      weekTickets,
      monthTickets,
      avgTicketsPerDay: Math.round(avgTicketsPerDay * 100) / 100,
    };

    const systemAnalysis = {
      ticketStats: {
        totalTickets,
        openTickets,
        inProgressTickets,
        completedTickets,
        closedTickets,
        averageResolutionTime: Math.round(averageResolutionTime * 100) / 100,
      },
      departmentStats,
      severityStats,
      technicianStats,
      timeStats,
    };

    return res.status(200).json({ ok: true, ...systemAnalysis });
  } catch (error) {
    console.error("Error fetching system analysis:", error);
    return res
      .status(500)
      .json({ ok: false, error: "Failed to fetch system analysis" });
  }
}
